/**
  ******************************************************************************
  * @file    py32f410_hal_msp.c
  * @author  MCU Application Team
  * @brief   This file provides code for the MSP Initialization
  *          and de-Initialization codes.
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2016 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "main.h"

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* External functions --------------------------------------------------------*/

/**
  * @brief Initialize global MSP
  */
void HAL_MspInit(void)
{
  __HAL_RCC_SYSCFG_CLK_ENABLE();
  __HAL_RCC_PWR_CLK_ENABLE();
}

#ifdef TRANSMITTER_BOARD
/**
* @brief USART MSP Initialization
* This function configures the hardware resources used in this example
* @param husart: USART handle pointer
* @retval None
*/
void HAL_USART_MspInit(USART_HandleTypeDef* husart)
{
  GPIO_InitTypeDef GPIO_InitStruct = {0};

  /* Peripheral clock enable */
  __HAL_RCC_USART1_CLK_ENABLE();
  __HAL_RCC_GPIOA_CLK_ENABLE();
  __HAL_RCC_DMA1_CLK_ENABLE();
  
  /**USART1 GPIO Configuration
  PA08    ------> USART1_CK
  PA09    ------> USART1_TX
  PA10    ------> USART1_RX
  */
  GPIO_InitStruct.Pin       = GPIO_PIN_9 | GPIO_PIN_10;
  GPIO_InitStruct.Mode      = GPIO_MODE_AF_PP;
  GPIO_InitStruct.Pull      = GPIO_PULLDOWN;
  GPIO_InitStruct.Speed     = GPIO_SPEED_FREQ_VERY_HIGH;
  GPIO_InitStruct.Alternate = GPIO_AF13_USART1;
  HAL_GPIO_Init(GPIOA, &GPIO_InitStruct);
  
  GPIO_InitStruct.Pin       = GPIO_PIN_8;
  GPIO_InitStruct.Alternate = GPIO_AF14_USART1;
  HAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

  /* USART1 DMA Init */
  /* USART1_RX Init */
  hdma_usart_rx.Instance                 = DMA1_Channel2;
  hdma_usart_rx.Init.Direction           = DMA_PERIPH_TO_MEMORY;
  hdma_usart_rx.Init.PeriphInc           = DMA_PADDR_FIX;
  hdma_usart_rx.Init.MemInc              = DMA_MADDR_INC;
  hdma_usart_rx.Init.PeriphDataAlignment = DMA_PDATAALIGN_BYTE;
  hdma_usart_rx.Init.MemDataAlignment    = DMA_MDATAALIGN_BYTE;
  hdma_usart_rx.Init.Mode                = DMA_NORMAL;
  hdma_usart_rx.Init.Priority            = DMA_PRIORITY_1;
  hdma_usart_rx.Init.TransMode           = DMA_TRANSFER_SINGLE;
  /* DMA initialization */
  if (HAL_DMA_Init(&hdma_usart_rx) != HAL_OK)
  {
    APP_ErrorHandler();
  }
  HAL_DMA_ChannelMap(&hdma_usart_rx, DMA_CHANNEL_MAP_USART_RX);
  __HAL_LINKDMA(husart, hdmarx, hdma_usart_rx);

  /* USART1_TX Init */
  hdma_usart_tx.Instance                 = DMA1_Channel1;
  hdma_usart_tx.Init.Direction           = DMA_MEMORY_TO_PERIPH;
  hdma_usart_tx.Init.PeriphInc           = DMA_PADDR_FIX;
  hdma_usart_tx.Init.MemInc              = DMA_MADDR_INC;
  hdma_usart_tx.Init.PeriphDataAlignment = DMA_PDATAALIGN_BYTE;
  hdma_usart_tx.Init.MemDataAlignment    = DMA_MDATAALIGN_BYTE;
  hdma_usart_tx.Init.Mode                = DMA_NORMAL;
  hdma_usart_tx.Init.Priority            = DMA_PRIORITY_0;
  hdma_usart_tx.Init.TransMode           = DMA_TRANSFER_SINGLE;

  if (HAL_DMA_Init(&hdma_usart_tx) != HAL_OK)
  {
    APP_ErrorHandler();
  }
  HAL_DMA_ChannelMap(&hdma_usart_tx, DMA_CHANNEL_MAP_USART_TX);
  __HAL_LINKDMA(husart,hdmatx,hdma_usart_tx);

  /* NVIC enable */
  HAL_NVIC_SetPriority(DMA1_Channel1_IRQn, 2, 0);
  HAL_NVIC_EnableIRQ(DMA1_Channel1_IRQn);
  HAL_NVIC_SetPriority(DMA1_Channel2_IRQn, 2, 0);
  HAL_NVIC_EnableIRQ(DMA1_Channel2_IRQn);
  /* USART interrupt Init */
  HAL_NVIC_SetPriority(USART_IRQn, 1, 0);
  HAL_NVIC_EnableIRQ(USART_IRQn);
}
#else 
/**
* @brief SPI MSP Initialization
* This function configures the hardware resources used in this example
* @param hspi: SPI handle pointer
* @retval None
*/
void HAL_SPI_MspInit(SPI_HandleTypeDef *hspi)
{
  GPIO_InitTypeDef GPIO_InitStruct = {0};

  /* Peripheral clock enable */
  __HAL_RCC_SPI1_CLK_ENABLE();                    /* SPI1 clock enable */
  __HAL_RCC_GPIOA_CLK_ENABLE();                   /* GPIOA clock enable */
  __HAL_RCC_DMA1_CLK_ENABLE();                    /* DMA clock enable */
  
  /**SPI1 GPIO Configuration
    PA5   ------> SCK
    PA6   ------> MISO
    PA7   ------> MOSI
  */
  /* SPI SCK GPIO pin configuration  */
  GPIO_InitStruct.Pin       = GPIO_PIN_5;
  if (hspi->Init.CLKPolarity == SPI_POLARITY_LOW)
  {
    GPIO_InitStruct.Pull = GPIO_PULLDOWN;
  }
  else
  {
    GPIO_InitStruct.Pull = GPIO_PULLUP;
  }
  GPIO_InitStruct.Mode      = GPIO_MODE_AF_PP;
  GPIO_InitStruct.Speed     = GPIO_SPEED_FREQ_HIGH;
  GPIO_InitStruct.Alternate = GPIO_AF1_SPI1;
  HAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

  /* Configure GPIO as SPI: MISO/MOSI*/
  GPIO_InitStruct.Pin       = GPIO_PIN_6 | GPIO_PIN_7;
  GPIO_InitStruct.Mode      = GPIO_MODE_AF_PP;
  GPIO_InitStruct.Pull      = GPIO_PULLDOWN;
  GPIO_InitStruct.Speed     = GPIO_SPEED_FREQ_HIGH;
  GPIO_InitStruct.Alternate = GPIO_AF1_SPI1;
  HAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

  /* SPI1 DMA Init */
  /* DMA_CH1 configuration */
  hdma_spi_tx.Instance                   = DMA1_Channel1;
  hdma_spi_tx.Init.Direction             = DMA_MEMORY_TO_PERIPH;
  hdma_spi_tx.Init.PeriphInc             = DMA_PADDR_FIX;
  hdma_spi_tx.Init.MemInc                = DMA_MADDR_INC;
  if (Spi1Handle.Init.DataSize <= SPI_DATASIZE_8BIT)
  {
    hdma_spi_tx.Init.PeriphDataAlignment = DMA_PDATAALIGN_BYTE;
    hdma_spi_tx.Init.MemDataAlignment    = DMA_MDATAALIGN_BYTE;
  }
  else
  {
    hdma_spi_tx.Init.PeriphDataAlignment = DMA_PDATAALIGN_HALFWORD;
    hdma_spi_tx.Init.MemDataAlignment    = DMA_MDATAALIGN_HALFWORD;
  }
  hdma_spi_tx.Init.Mode                  = DMA_NORMAL;
  hdma_spi_tx.Init.Priority              = DMA_PRIORITY_1;
  hdma_spi_tx.Init.TransMode             = DMA_TRANSFER_SINGLE;
  /* Initialize DMA */
  HAL_DMA_Init(&hdma_spi_tx);
  
  /* Set DMA channel map. */
  HAL_DMA_ChannelMap(&hdma_spi_tx, DMA_CHANNEL_MAP_SPI1_TX);
  /* DMA handle is associated with SPI handle */
  __HAL_LINKDMA(&Spi1Handle, hdmatx, hdma_spi_tx);

  /* DMA_CH2 configuration */
  hdma_spi_rx.Instance                   = DMA1_Channel2;
  hdma_spi_rx.Init.Direction             = DMA_PERIPH_TO_MEMORY;
  hdma_spi_rx.Init.PeriphInc             = DMA_PADDR_FIX;
  hdma_spi_rx.Init.MemInc                = DMA_MADDR_INC;
  if (Spi1Handle.Init.DataSize <= SPI_DATASIZE_8BIT)
  {
    hdma_spi_rx.Init.PeriphDataAlignment = DMA_PDATAALIGN_BYTE;
    hdma_spi_rx.Init.MemDataAlignment    = DMA_MDATAALIGN_BYTE;
  }
  else
  {
    hdma_spi_rx.Init.PeriphDataAlignment = DMA_PDATAALIGN_HALFWORD;
    hdma_spi_rx.Init.MemDataAlignment    = DMA_MDATAALIGN_HALFWORD;
  }
  hdma_spi_rx.Init.Mode                  = DMA_NORMAL;
  hdma_spi_rx.Init.Priority              = DMA_PRIORITY_0;
  hdma_spi_rx.Init.TransMode             = DMA_TRANSFER_SINGLE;
  
  /* Initialize DMA*/
  HAL_DMA_Init(&hdma_spi_rx);
  
  /* Set DMA channel map. */
  HAL_DMA_ChannelMap(&hdma_spi_rx, DMA_CHANNEL_MAP_SPI1_RX);
  /* DMA handle is associated with SPI handle */
  __HAL_LINKDMA(&Spi1Handle, hdmarx, hdma_spi_rx);

  /* NVIC enable */
  HAL_NVIC_SetPriority(DMA1_Channel1_IRQn, 2, 0);
  HAL_NVIC_EnableIRQ(DMA1_Channel1_IRQn);
  HAL_NVIC_SetPriority(DMA1_Channel2_IRQn, 2, 0);
  HAL_NVIC_EnableIRQ(DMA1_Channel2_IRQn);
  /* Interrupt configuration */
  HAL_NVIC_SetPriority(SPI1_IRQn, 1, 0);
  HAL_NVIC_EnableIRQ(SPI1_IRQn);
}
#endif

/************************ (C) COPYRIGHT Puya *****END OF FILE******************/
